const entityData = [
	{
		"character": "&#128125;",
		"htmlEntityName": "&nbsp;",
		"htmlEntityNumber": "&#160;",
		"description": "Non-breaking space"
	},
	{
		"character": "&#128760;",
		"htmlEntityName": "&zwnj;",
		"htmlEntityNumber": "&#8204;",
		"description": "Zero width non-joiner"
	},
	{
		"character": "👾",
		"htmlEntityName": "&#65279;",
		"htmlEntityNumber": "&#65279;",
		"description": "Zero width no-break space" //(dates & address blue links)
	},
	{
		"character": "🧟",
		"htmlEntityName": "&zwj;",
		"htmlEntityNumber": "&#8205;",
		"description": "Zero width joiner"
	},
	{
		"character": "🪐",
		"htmlEntityName": "&#847;",
		"htmlEntityNumber": "&#847;",
		"description": "Combining grapheme joiner"
	},
	{
		"character": "🔮",
		"htmlEntityName": "&#8203;",
		"htmlEntityNumber": "&#8203;",
		"description": "Zero width space"
	},
	{
		"character": "&",
		"htmlEntityName": "&amp;",
		"htmlEntityNumber": "&#38;",
		"description": "Ampersand"
	},
	{
		"character": "#",
		"htmlEntityName": "&num;",
		"htmlEntityNumber": "&#35;",
		"description": "Number sign"
	},
	{
		"character": ";",
		"htmlEntityName": "&semi;",
		"htmlEntityNumber": "&#59;",
		"description": "Semicolon"
	},
	{
		"character": "‑",
		"htmlEntityName": "&#8209;",
		"htmlEntityNumber": "&#8209;",
		"description": "Non-breaking hyphen"
	},
	{
		"character": "–",
		"htmlEntityName": "&ndash;",
		"htmlEntityNumber": "&#150;",
		"description": "En dash &#8211;"
	},
	{
		"character": "—",
		"htmlEntityName": "&mdash;",
		"htmlEntityNumber": "&#151;",
		"description": "Em dash &#8212;"
	},
	{
		"character": "-",
		"htmlEntityName": "&dash;",
		"htmlEntityNumber": "&#8208;",
		"description": "Hyphen or dash"
	},
	{
		"character": "“",
		"htmlEntityName": "&ldquo;",
		"htmlEntityNumber": "&#147;",
		"description": "Left double quote &#8220;"
	},
	{
		"character": "”",
		"htmlEntityName": "&rdquo;",
		"htmlEntityNumber": "&#148;",
		"description": "Right double quote &#8221;"
	},
	{
		"character": "‘",
		"htmlEntityName": "&lsquo;",
		"htmlEntityNumber": "&#145;",
		"description": "Left single quote &#8216;"
	},
	{
		"character": "’",
		"htmlEntityName": "&rsquo;",
		"htmlEntityNumber": "&#146;",
		"description": "Right single quote &#8217;"
	},
	{
		"character": "'",
		"htmlEntityName": "&apos;",
		"htmlEntityNumber": "&#39;",
		"description": "Single quote (apostrophe)"
	},
	{
		"character": "&quot;",
		"htmlEntityName": "&quot;",
		"htmlEntityNumber": "&#34;",
		"description": "Quotation mark"
	},
	{
		"character": "<",
		"htmlEntityName": "&lt;",
		"htmlEntityNumber": "&#60;",
		"description": "Less than sign"
	},
	{
		"character": ">",
		"htmlEntityName": "&gt;",
		"htmlEntityNumber": "&#62;",
		"description": "Greater than sign"
	},
	{
		"character": "©",
		"htmlEntityName": "&copy;",
		"htmlEntityNumber": "&#169;",
		"description": "Copyright symbol"
	},
	{
		"character": "®",
		"htmlEntityName": "&reg;",
		"htmlEntityNumber": "&#174;",
		"description": "Registered trademark symbol"
	},
	{
		"character": "™",
		"htmlEntityName": "&trade;",
		"htmlEntityNumber": "&#153;",
		"description": "Trademark symbol &#8482;"
	},
	{
		"character": "€",
		"htmlEntityName": "&euro;",
		"htmlEntityNumber": "&#128;",
		"description": "Euro currency symbol &#8364;"
	},
	{
		"character": "¥",
		"htmlEntityName": "&yen;",
		"htmlEntityNumber": "&#165;",
		"description": "Yen currency symbol"
	},
	{
		"character": "£",
		"htmlEntityName": "&pound;",
		"htmlEntityNumber": "&#163;",
		"description": "Pound sterling symbol"
	},
	{
		"character": "¢",
		"htmlEntityName": "&cent;",
		"htmlEntityNumber": "&#162;",
		"description": "Cent symbol"
	},
	{
		"character": "-",
		"htmlEntityName": "&minus;",
		"htmlEntityNumber": "&#8722;",
		"description": "Minus sign"
	},
	{
		"character": "…",
		"htmlEntityName": "&hellip;",
		"htmlEntityNumber": "&#8230;",
		"description": "Ellipsis"
	},
	{
		"character": "•",
		"htmlEntityName": "&bull;",
		"htmlEntityNumber": "&#8226;",
		"description": "Bullet point"
	},
	{
		"character": "•",
		"htmlEntityName": "&middot;",
		"htmlEntityNumber": "&#183;",
		"description": "Middle dot"
	},
	{
		"character": "×",
		"htmlEntityName": "&times;",
		"htmlEntityNumber": "&#215;",
		"description": "Multiplication sign"
	},
	{
		"character": "÷",
		"htmlEntityName": "&divide;",
		"htmlEntityNumber": "&#247;",
		"description": "Division sign"
	},
	{
		"character": "±",
		"htmlEntityName": "&plusmn;",
		"htmlEntityNumber": "&#177;",
		"description": "Plus-minus sign"
	},
	{
		"character": "+",
		"htmlEntityName": "&plus;",
		"htmlEntityNumber": "&#43;",
		"description": "Plus sign"
	},
	{
		"character": "°",
		"htmlEntityName": "&deg;",
		"htmlEntityNumber": "&#176;",
		"description": "Degree symbol"
	},
	{
		"character": "‰",
		"htmlEntityName": "&permil;",
		"htmlEntityNumber": "&#8240;",
		"description": "Per mille sign"
	},
	{
		"character": "℠",
		"htmlEntityName": "&#8480;",
		"htmlEntityNumber": "&#8480;",
		"description": "SM Service mark"
	},
	{
		"character": "†",
		"htmlEntityName": "&dagger;",
		"htmlEntityNumber": "&#134;",
		"description": "Dagger symbol &#8224;"
	},
	{
		"character": "‡",
		"htmlEntityName": "&Dagger;",
		"htmlEntityNumber": "&#135;",
		"description": "Double dagger symbol &#8225;"
	},
	{
		"character": "§",
		"htmlEntityName": "&sect;",
		"htmlEntityNumber": "&#167;",
		"description": "Section sign"
	},
	{
		"character": "»",
		"htmlEntityName": "&raquo;",
		"htmlEntityNumber": "&#187;",
		"description": "Right pointing double angle quotation mark"
	},
	{
		"character": "«",
		"htmlEntityName": "&laquo;",
		"htmlEntityNumber": "&#171;",
		"description": "Left pointing double angle quotation mark"
	},
	{
		"character": "―",
		"htmlEntityName": "&horbar;",
		"htmlEntityNumber": "&#8213;",
		"description": "Horizontal bar"
	},
	{
		"character": "~",
		"htmlEntityName": "&tilde;",
		"htmlEntityNumber": "&#126;",
		"description": "Tilde"
	},
	{
		"character": "!",
		"htmlEntityName": "&excl;",
		"htmlEntityNumber": "&#33;",
		"description": "Exclamation mark"
	},
	{
		"character": "@",
		"htmlEntityName": "&commat;",
		"htmlEntityNumber": "&#64;",
		"description": "At symbol"
	},
	{
		"character": "$",
		"htmlEntityName": "&dollar;",
		"htmlEntityNumber": "&#36;",
		"description": "Dollar sign"
	},
	{
		"character": "%",
		"htmlEntityName": "&percnt;",
		"htmlEntityNumber": "&#37;",
		"description": "Percent sign"
	},
	{
		"character": "^",
		"htmlEntityName": "&Hat;",
		"htmlEntityNumber": "&#94;",
		"description": "Circumflex accent"
	},
	{
		"character": "*",
		"htmlEntityName": "&ast;",
		"htmlEntityNumber": "&#42;",
		"description": "Asterisk &#9733;"
	},
	{
		"character": "~",
		"htmlEntityName": "&#126;",
		"htmlEntityNumber": "&#126;",
		"description": "Tilde"
	},
	{
		"character": "`",
		"htmlEntityName": "&#96;",
		"htmlEntityNumber": "&DiacriticalGrave;",
		"description": "Grave accent"
	},
	{
		"character": "[",
		"htmlEntityName": "&lbrack;",
		"htmlEntityNumber": "&#91;",
		"description": "Left square bracket"
	},
	{
		"character": "]",
		"htmlEntityName": "&rbrack;",
		"htmlEntityNumber": "&#93;",
		"description": "Right square bracket"
	},
	{
		"character": "{",
		"htmlEntityName": "&lbrace;",
		"htmlEntityNumber": "&#123;",
		"description": "Left curly bracket"
	},
	{
		"character": "}",
		"htmlEntityName": "&rbrace;",
		"htmlEntityNumber": "&#125;",
		"description": "Right curly bracket"
	},
	{
		"character": "(",
		"htmlEntityName": "&lpar;",
		"htmlEntityNumber": "&#40;",
		"description": "Left parenthesis"
	},
	{
		"character": ")",
		"htmlEntityName": "&rpar;",
		"htmlEntityNumber": "&#41;",
		"description": "Right parenthesis"
	},
	{
		"character": "|",
		"htmlEntityName": "&vert;",
		"htmlEntityNumber": "&#124;",
		"description": "Vertical line &VerticalLine;"
	},
	{
		"character": "/",
		"htmlEntityName": "&sol;",
		"htmlEntityNumber": "&#47;",
		"description": "Forward slash solidus"
	},
	{
		"character": "\\",
		"htmlEntityName": "&bsol;",
		"htmlEntityNumber": "&#92;",
		"description": "Back slash reverse solidus"
	},
	{
		"character": "?",
		"htmlEntityName": "&quest;",
		"htmlEntityNumber": "&#63;",
		"description": "Question mark"
	},
	{
		"character": "¹",
		"htmlEntityName": "&sup1;",
		"htmlEntityNumber": "&#185;",
		"description": "Superscript one 1"
	},
	{
		"character": "²",
		"htmlEntityName": "&sup2;",
		"htmlEntityNumber": "&#178;",
		"description": "Superscript two 2"
	},
	{
		"character": "³",
		"htmlEntityName": "&sup3;",
		"htmlEntityNumber": "&#179;",
		"description": "Superscript three 3"
	},
	{
		"character": "µ",
		"htmlEntityName": "&micro;",
		"htmlEntityNumber": "&#181;",
		"description": "Micro sign"
	},
	{
		"character": "¶",
		"htmlEntityName": "&para;",
		"htmlEntityNumber": "&#182;",
		"description": "Paragraph sign"
	},
	{
		"character": "_",
		"htmlEntityName": "&lowbar;",
		"htmlEntityNumber": "&#95;",
		"description": "Underscore"
	},
	{
		"character": ":",
		"htmlEntityName": "&colon;",
		"htmlEntityNumber": "&#58;",
		"description": "Colon"
	},

	{
		"character": "á",
		"htmlEntityName": "&aacute;",
		"htmlEntityNumber": "&#225;",
		"description": "Latin small letter a with acute"
	},
	{
		"character": "é",
		"htmlEntityName": "&eacute;",
		"htmlEntityNumber": "&#233;",
		"description": "Latin small letter e with acute"
	},
	{
		"character": "ë",
		"htmlEntityName": "&euml;",
		"htmlEntityNumber": "&#235;",
		"description": "Latin small letter e diaeresis"
	},
	{
		"character": "í",
		"htmlEntityName": "&iacute;",
		"htmlEntityNumber": "&#237;",
		"description": "Latin small letter i with acute"
	},
	{
		"character": "î",
		"htmlEntityName": "&icirc;",
		"htmlEntityNumber": "&#238;",
		"description": "Latin small letter i circumflex"
	},
	{
		"character": "ï",
		"htmlEntityName": "&iuml;",
		"htmlEntityNumber": "&#239;",
		"description": "Latin small letter i diaeresis"
	},
	{
		"character": "ó",
		"htmlEntityName": "&oacute;",
		"htmlEntityNumber": "&#243;",
		"description": "Latin small letter o with acute"
	},
	{
		"character": "ú",
		"htmlEntityName": "&uacute;",
		"htmlEntityNumber": "&#250;",
		"description": "Latin small letter U with acute"
	},
	{
		"character": "ñ",
		"htmlEntityName": "&ntilde;",
		"htmlEntityNumber": "&#241;",
		"description": "Latin small letter N with tilde"
	},
	{
		"character": "Á",
		"htmlEntityName": "&Aacute;",
		"htmlEntityNumber": "&#193;",
		"description": "Latin capital letter A with acute"
	},
	{
		"character": "É",
		"htmlEntityName": "&Eacute;",
		"htmlEntityNumber": "&#201;",
		"description": "Latin capital letter E with acute"
	},
	{
		"character": "Í",
		"htmlEntityName": "&Iacute;",
		"htmlEntityNumber": "&#205;",
		"description": "Latin capital letter I with acute"
	},
	{
		"character": "Ó",
		"htmlEntityName": "&Oacute;",
		"htmlEntityNumber": "&#211;",
		"description": "Latin capital letter O with acute"
	},
	{
		"character": "Ú",
		"htmlEntityName": "&Uacute;",
		"htmlEntityNumber": "&#218;",
		"description": "Latin capital letter U with acute"
	},
	{
		"character": "Ñ",
		"htmlEntityName": "&Ntilde;",
		"htmlEntityNumber": "&#209;",
		"description": "Latin capital letter N with tilde"
	},
	{
		"character": "¿",
		"htmlEntityName": "&iquest;",
		"htmlEntityNumber": "&#191;",
		"description": "Inverted question mark"
	},
	{
		"character": "¡",
		"htmlEntityName": "&iexcl;",
		"htmlEntityNumber": "&#161;",
		"description": "Inverted exclamation mark"
	},

	{
		"character": "ä",
		"htmlEntityName": "&auml;",
		"htmlEntityNumber": "&#228;",
		"description": "Latin Small Letter A with Diaeresis (Umlaut)"
	},
	{
		"character": "ö",
		"htmlEntityName": "&ouml;",
		"htmlEntityNumber": "&#246;",
		"description": "Latin Small Letter O with Diaeresis (Umlaut)"
	},
	{
		"character": "ü",
		"htmlEntityName": "&uuml;",
		"htmlEntityNumber": "&#252;",
		"description": "Latin Small Letter U with Diaeresis (Umlaut)"
	},
	{
		"character": "ß",
		"htmlEntityName": "&szlig;",
		"htmlEntityNumber": "&#223;",
		"description": "Latin Small Letter Sharp S (Eszett)"
	},
	{
		"character": "Ä",
		"htmlEntityName": "&Auml;",
		"htmlEntityNumber": "&#196;",
		"description": "Latin Capital Letter A with Diaeresis (Umlaut)"
	},
	{
		"character": "Ö",
		"htmlEntityName": "&Ouml;",
		"htmlEntityNumber": "&#214;",
		"description": "Latin Capital Letter O with Diaeresis (Umlaut)"
	},
	{
		"character": "Ü",
		"htmlEntityName": "&Uuml;",
		"htmlEntityNumber": "&#220;",
		"description": "Latin Capital Letter U with Diaeresis (Umlaut)"
	},
	{
		"character": "Ü",
		"htmlEntityName": "&Uuml;",
		"htmlEntityNumber": "&#220;",
		"description": "Latin Capital Letter U with Diaeresis (Umlaut)"
	},
	{
		"character": "ç",
		"htmlEntityName": "&ccedil;",
		"htmlEntityNumber": "&#231;",
		"description": "Latin small letter c cedilla"
	},
	{
		"character": "Ç",
		"htmlEntityName": "&Ccedil;",
		"htmlEntityNumber": "&#199;",
		"description": "Latin capital letter c cedilla"
	},
	{
		"character": "`",
		"htmlEntityName": "&grave;",
		"htmlEntityNumber": "&#96;",
		"description": "Grave Accent"
	},
	{
		"character": "´",
		"htmlEntityName": "&acute;",
		"htmlEntityNumber": "&#180;",
		"description": "Acute Accent"
	},

];


const entities = entityData;
let currentEntities = entities;

const entitiesSearchInput = document.getElementById('entitiesSearchInput');
const entitySearchResults = document.getElementById('entitySearchResults');
const entityResultCount = document.getElementById('entityResultCount');
const noSpecialCharResults = document.getElementById('noSpecialCharResults');

const columnsPerRow = 4;
let maxColumns = 12;

function searchEntities() {
    const searchTerm = entitiesSearchInput.value.trim().toLowerCase();

    if (searchTerm.startsWith("u+")) {
        const unicodeCodePoint = searchTerm.substring(2);
        currentEntities = entities.filter(entity => (entity.unicode || "").toLowerCase() === unicodeCodePoint);
    } else if (searchTerm === "") {
        currentEntities = entities;
    } else {
        const searchTerms = searchTerm.split(/\s+/).filter(term => term !== "");
        currentEntities = entities.filter(entity => {
            return searchTerms.every(term => {
                return (
                    (entity.character || "").toLowerCase().includes(term) ||
                    (entity.description || "").toLowerCase().includes(term) ||
                    (entity.htmlEntityName || "").toLowerCase().includes(term) ||
                    (entity.htmlEntityNumber || "").toLowerCase().includes(term)
                );
            });
        });
    }

    // Function to toggle the visibility of the noSpecialCharResults div
    if (currentEntities.length === 0) {
        entitySearchResults.style.display = 'none';
        noSpecialCharResults.style.display = 'block';
    } else {
        entitySearchResults.style.display = 'flex';
        noSpecialCharResults.style.display = 'none';
    }

    displayEntities(currentEntities);
    updateEntityResultCount(currentEntities.length);
}

function displayEntities(entitiesToDisplay) {
    entitySearchResults.innerHTML = '';
    let displayedEntities = entitiesToDisplay;

    if (displayedEntities.length > maxColumns) {
        displayedEntities = displayedEntities.slice(0, maxColumns);
    }

    for (let i = 0; i < Math.ceil(displayedEntities.length / columnsPerRow); i++) {
        for (let j = 0; j < columnsPerRow && i * columnsPerRow + j < displayedEntities.length; j++) {
            const entity = displayedEntities[i * columnsPerRow + j];
            const column = document.createElement('div');
            column.classList.add('col-sm-6', 'col-md-4', 'col-lg-3', 'p-2');

            const nameContainer = document.createElement('pre');
            nameContainer.textContent = entity.htmlEntityName || "";
            nameContainer.classList.add("entity-name", "copy-target");

            const numberContainer = document.createElement('pre');
            numberContainer.textContent = entity.htmlEntityNumber || "";
            numberContainer.classList.add("entity-number", "copy-target");

            const descriptionContainer = document.createElement('div');
            descriptionContainer.classList.add("entity-description");
            descriptionContainer.textContent = entity.description || "";

            column.innerHTML = `
                <div class="entity-column">
                    <p class="entity-character">${entity.character || ""}</p>
                    <div class="entity-details">
                    </div>
                </div>
            `;

            const detailsContainer = column.querySelector('.entity-details');
            detailsContainer.appendChild(nameContainer);
            detailsContainer.appendChild(numberContainer);
            detailsContainer.appendChild(descriptionContainer);

            entitySearchResults.appendChild(column);
        }
    }
    addEntityClickToCopyListeners();
}

function addEntityClickToCopyListeners() {
    const copyTargets = document.querySelectorAll(".copy-target, .entity-character");

    copyTargets.forEach(target => {
        target.addEventListener("click", () => {
            let textToCopy = target.textContent;
            if (target.classList.contains("entity-character")) {
                const parent = target.parentNode;
                const detailsContainer = parent.querySelector('.entity-details');
                const nameElement = detailsContainer.querySelector('.entity-name');
                if (nameElement) {
                    const name = nameElement.textContent;
                    const correctEntity = entityData.find(entity => entity.htmlEntityName === name);
                    if (correctEntity) {
                        textToCopy = correctEntity.character;
                    }
                }
            }
            navigator.clipboard.writeText(textToCopy)
                .then(() => {
                    target.classList.add("copied");
                    setTimeout(() => target.classList.remove("copied"), 1100);
                })
                .catch(err => console.error("Failed to copy text:", err));
        });
    });
}

function updateEntityResultCount(count) {
    entityResultCount.textContent = `${count} ${count === 1 ? 'result' : 'results'}`;
}

// Show All/Show Less Button
showAllButton.addEventListener('click', function() {
	if (maxColumns === 12) {
		 maxColumns = 200;
		 this.textContent = 'Show Less';
	} else {
		 maxColumns = 12;
		 this.textContent = 'Show All';
	}
	displayEntities(currentEntities); // Re-render with the new maxColumns
});



// Initial display and count are now inside the event listener
entitiesSearchInput.addEventListener('input', searchEntities);
searchEntities();
