const todoInput = document.querySelector(".todo-input");
const todoButton = document.querySelector(".todo-button");
const todoList = document.querySelector(".todo-list");
const allBtn = document.querySelector('.all-btn');
const completedBtn = document.querySelector('.completed-btn');
const uncompletedBtn = document.querySelector('.uncompleted-btn');

let currentFilter = "all";

setActiveButton(allBtn);

document.addEventListener("DOMContentLoaded", getTodos);
todoButton.addEventListener("click", addTodo);
todoList.addEventListener("click", deleteCheck);

allBtn.addEventListener('click', (event) => handleFilterClick(event, "all", allBtn));
completedBtn.addEventListener('click', (event) => handleFilterClick(event, "completed", completedBtn));
uncompletedBtn.addEventListener('click', (event) => handleFilterClick(event, "uncompleted", uncompletedBtn));

function handleFilterClick(event, filter, button) {
    event.preventDefault();
    currentFilter = filter;
    setActiveButton(button);
    filterTodos();
    localStorage.setItem('todoFilter', filter); // Save All/Not Completed/Completed filter in cache
}

function setActiveButton(button) {
    allBtn.classList.remove('active');
    completedBtn.classList.remove('active');
    uncompletedBtn.classList.remove('active');
    button.classList.add('active');
}

function addTodo(event) {
    event.preventDefault();

    const todoText = todoInput.value.trim(); // Trim whitespace directly
    if (!todoText) return; // More concise empty check

    const todoId = Date.now().toString();
    const todo = { id: todoId, text: todoText, completed: false };

    const todoDiv = createTodoElement(todo); // Use helper function
    todoList.appendChild(todoDiv);
    saveLocalTodo(todo);

    todoInput.value = "";
    filterTodos();
}

function createTodoElement(todo) { // Helper function to create todo elements
    const todoDiv = document.createElement("div");
    todoDiv.classList.add("todo");
    todoDiv.dataset.id = todo.id;
    if (todo.completed) {
        todoDiv.classList.add("completed");
    }

    const newTodo = document.createElement("div");
    newTodo.innerText = todo.text;
    newTodo.classList.add("todo-item");
    todoDiv.appendChild(newTodo);

    const completedButton = document.createElement("button");
    completedButton.innerHTML = '<i class="bi bi-check-lg"></i>';
    completedButton.classList.add("complete-btn");
    todoDiv.appendChild(completedButton);

    const trashButton = document.createElement("button");
    trashButton.innerHTML = '<i class="bi bi-trash3"></i>';
    trashButton.classList.add("trash-btn");
    todoDiv.appendChild(trashButton);

    return todoDiv;
}

function deleteCheck(e) {
    const item = e.target;
    const todo = item.closest(".todo");
    if (!todo) return;

    const todoId = todo.dataset.id;

    if (item.classList.contains('complete-btn') || item.parentElement.classList.contains('complete-btn')) {
        todo.classList.toggle("completed");
        updateLocalTodo(todoId, todo.classList.contains("completed"));
    } else if (item.classList.contains('trash-btn') || item.parentElement.classList.contains('trash-btn')) {
        removeLocalTodo(todoId);
        todo.remove();
    }
    filterTodos(); // Filter outside the if/else to avoid repetition
}

function filterTodo(filter) {
    const todos = Array.from(todoList.children);
    todos.forEach(todo => {
        todo.style.display = (filter === "all" ||
                             (filter === "completed" && todo.classList.contains("completed")) ||
                             (filter === "uncompleted" && !todo.classList.contains("completed")))
                             ? "flex" : "none"; // More concise conditional
    });
}

function filterTodos() {
    filterTodo(currentFilter);
}

function saveLocalTodo(todo) {
    let todos = JSON.parse(localStorage.getItem('todos')) || [];
    todos.push(todo);
    localStorage.setItem('todos', JSON.stringify(todos));
}

function getTodos() {
    let todos = JSON.parse(localStorage.getItem('todos')) || [];

    //All/Not Completed/Completed filter
    const storedFilter = localStorage.getItem('todoFilter');
    if (storedFilter) {
        currentFilter = storedFilter;
        if (storedFilter === "completed"){
            setActiveButton(completedBtn);
        } else if (storedFilter === "uncompleted"){
            setActiveButton(uncompletedBtn);
        } else {
            setActiveButton(allBtn);
        }
    } else {
        setActiveButton(allBtn);
    } //End All/Not Completed/Completed filter

    todos.forEach(todo => {
        if (todo && todo.id && todo.text !== undefined && todo.completed !== undefined) {
            const todoDiv = createTodoElement(todo); // Use helper function
            todoList.appendChild(todoDiv);
        } else {
            console.error("Malformed todo item found in local storage:", todo);
            // removeMalformedTodo(todo); // Uncomment if you want to remove malformed todos
        }
    });
    filterTodos();
}

function removeLocalTodo(todoId) {
    let todos = JSON.parse(localStorage.getItem('todos')) || [];
    todos = todos.filter(todo => todo.id !== todoId);
    localStorage.setItem("todos", JSON.stringify(todos));
}

function updateLocalTodo(todoId, completed) {
    let todos = JSON.parse(localStorage.getItem('todos')) || [];
    todos = todos.map(todo => todo.id === todoId ? { ...todo, completed } : todo); // More concise map
    localStorage.setItem("todos", JSON.stringify(todos));
}

function removeMalformedTodo(malformedTodo) {
    let todos = JSON.parse(localStorage.getItem('todos')) || [];
    todos = todos.filter(todo => todo !== malformedTodo);
    localStorage.setItem("todos", JSON.stringify(todos));
}


// Focus Styling for Form
const todoInputContainerStyling = document.querySelector('.todo-input-container');
const todoInputStyling = document.querySelector('.todo-input');

todoInputStyling.addEventListener('focus', () => {
  todoInputContainerStyling.classList.add('focused');
});

todoInputStyling.addEventListener('blur', () => {
  todoInputContainerStyling.classList.remove('focused');
});


// SORTABLE.JS WITH ORDER CACHING
var el = document.getElementById('sortable-todo');

// Function to load the sorted order from local storage
function loadSortedOrder() {
  const storedOrder = localStorage.getItem('sortedItems');
  if (storedOrder) {
    const sortedItems = JSON.parse(storedOrder);
    // Reorder the list based on the stored order
    const listItems = Array.from(el.children); // Get current list items
    sortedItems.forEach(itemText => {
      const listItem = listItems.find(li => li.textContent === itemText);
      if (listItem) {
        el.appendChild(listItem); // Move to correct position
      }
    });
  }
}

// Load the sorted order on page load
document.addEventListener('DOMContentLoaded', loadSortedOrder);

var sortable = Sortable.create(el, {
  animation: 150,
  ghostClass: 'sortable-ghost',
  chosenClass: 'sortable-chosen',
  dragClass: 'sortable-drag',
  onEnd: function (/**Event*/evt) {
    const sortedItems = Array.from(el.children).map(li => li.textContent);
    localStorage.setItem('sortedItems', JSON.stringify(sortedItems));
    console.log("New Order:", sortedItems);
  },
});